package com.easou.ps.common.pool;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.log4j.Logger;

public class ServicePool<T> {

	private static Logger POOL = Logger.getLogger("pool");
	private String poolName;
	private Factory<T> factory; 
	private int sizePreHost;
	private MaintThread maint = null;
	private AtomicInteger seq = new AtomicInteger(0);
	private List<ServiceHandle<T>> alive = new  CopyOnWriteArrayList<ServiceHandle<T>>();
	private Set<ServiceHandle<T>> dead = new  HashSet<ServiceHandle<T>>();
	private Map<String, ServiceHandle<T>> addr2Handle = new ConcurrentHashMap<String, ServiceHandle<T>>(); 

	public ServicePool (Factory<T> factory, int sizePreHost, String ... addresses){
		
		if (addresses == null || addresses.length ==0) {
			throw new IllegalArgumentException("service address is empty : " + addresses);
		}

		for (String address : addresses ){
			ServiceHandle<T> handle = new ServiceHandle<T>(address);
			factory.replenishChannel(handle, sizePreHost);
			alive.add(handle);
			addr2Handle.put(address, handle);
		}
		
		this.factory = factory;
		this.sizePreHost = sizePreHost;
		this.poolName = this.getClass().getSimpleName();
		maint = new MaintThread();
		maint.start();
	}
	
	/**
	 * random select.
	 * @return
	 */
	ServiceHandle<T> selectHandle() {
		int size = alive.size();
		if (size > 0){
			int seqNo = seq.incrementAndGet();
			seqNo = Math.abs(seqNo);
			int index = seqNo % size;
			if ( index <alive.size() ){
				try {
					return alive.get(index);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			
			try {
				return alive.get(0);
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else {
			System.out.println(this.getClass().getSimpleName() +" alive service is empty!");
			POOL.warn(this.getClass().getSimpleName() +" alive service is empty!");
		}
		
		return null;
	}
	
	public T selectChannel(){
		return selectChannel(alive.size(), null);//根据互备服务数量设定重试次数
	}
	
	/**
	 * 返回指定服务地址的客户端链接
	 * @param address
	 * @return
	 */
	public T selectChannel(String address){
		return selectChannel(1, address);
	}
	
	/**
	 * 选取服务链接
	 * @param tryTime 尝试次数
	 * @return 返回可用与读写的链接，无可用链接返回null
	 */
	public T selectChannel(int tryTime, String address){
		for (int i=0; i< tryTime; i++){
			
			ServiceHandle<T> handle = null;
			if (address ==null || (handle= addr2Handle.get(address)) == null ){
				handle = selectHandle();
			}
			
			if (handle!=null){
				T t = handle.getChannel();
				if (t == null){
					System.out.println(" get channel is null from " +handle );
					POOL.warn(this.getClass().getSimpleName() +" get null channel from " +handle);
					continue;
				}
				
				if (factory.validateChannel(t)){
					return t ;
				}else {
					factory.destoryChannel(t);
					handle.invalid(t);
				}
			}
		}
		return null; //无可用链接
	}
	
	@Override
	public String toString() {
		StringBuilder str = new StringBuilder();
		str.append(this.getClass().getSimpleName())
		.append("{alive:").append(alive)
		.append(", dead:");
		 try {
			str.append(dead);
		} catch (Exception e) {
//			e.printStackTrace();
		}
		 str.append("}");
		return str.toString();
	}
	
	public void  stopMaint(){
		if (maint != null){
			maint.setRuning(false);
		}
	}
	
	public void  startMaint(){
		stopMaint();
		maint = new MaintThread();
		maint.start();
	}
	
	/**
	 * this thread detect the status of all addresses of all services every DETECT_INTERVAL_MILLIS(1min).
	 * for each address in every detecting, the thread tries to connect to it, retries on fail at an interval for RETRY_INTERVAL_MILLIS 
	 * for at most RETRY_TIMES, and then set status (active or broken) for the address. if an address is detected broken, 
	 * a warning mail is sent to relevant people in case the delta that current millis minus LAST_SEND_MAIL_TIME is greater than 30min.
	 *
	 */
	class MaintThread extends Thread {
		
		private static final int DETECT_INTERVAL_MILLIS = 1 * 1000;	//1 sec
		private static final int RETRY_TIMES = 3;
		private static final int RETRY_INTERVAL_MILLIS = 100 ;	//100 ms
		private final String[] MAIL_RECIPIENTS = new String[] {"hunter_lu@staff.easou.com" };
		private static final int SEND_MAIL_INTERVAL = 30 * 60 * 1000;	//30min
		private long LAST_SEND_MAIL_TIME = 0;		//it is accessed by a single thread, so VOLATILE is not necessary
		private volatile boolean runing = true;

		MaintThread() {
			this.setName(poolName+"->MaintThread");
			this.setDaemon( true );
		}
		
		public void run() {
			while (runing) {
				POOL.debug("start detecting service pool status, service kinds : " +poolName);
				try {
					long deltaMailTime = System.currentTimeMillis() - LAST_SEND_MAIL_TIME;
					boolean mailSent = false;
					
					// 有效handle缓存，避免重复检查
					Set<ServiceHandle<T>> restoredSet = new HashSet<ServiceHandle<T>>();// 复活Handle
					// dead address check
					Iterator<ServiceHandle<T>> deadItr = dead.iterator();
					while (deadItr.hasNext()){
						ServiceHandle<T> handle = deadItr.next();
						if (validateHandle(handle)){
							cleanInvalidChannel(handle);
							factory.replenishChannel(handle, sizePreHost);// 补充channel
							alive.add(handle);	//　添加到有效队列
							deadItr.remove();	//    从无效队列移除
							restoredSet.add(handle);
						}else {
							mailSent = sendWarning(deltaMailTime, handle);
						}
					}
					
					// alive address check
					Iterator<ServiceHandle<T>> aliveItr = alive.iterator();
					while (aliveItr.hasNext()){
						ServiceHandle<T> handle = aliveItr.next();
						if (restoredSet.contains(handle)){
							continue;
						}
						
						if (validateHandle(handle)){
							cleanInvalidChannel(handle);
							factory.replenishChannel(handle, sizePreHost);// 补充channel
						} else {
							alive.remove(handle);	//    从有效队列移除
							dead.add(handle);		//　添加到无效队列
							factory.destoryHandle(handle);	//关闭channel
							
							mailSent = sendWarning(deltaMailTime, handle);//报警
						}
					}
					
					if (mailSent) {
						LAST_SEND_MAIL_TIME = System.currentTimeMillis();
					}
					Thread.sleep(DETECT_INTERVAL_MILLIS);
				} catch (InterruptedException ie) {
					POOL.error("service pool status detecting thread is interrupted.");
					break;
				} catch (Exception e) {
					POOL.error("service pool status detecting thread meeting an upexpected exception:" + e.getMessage(), e);
				}
			}
		}

		private boolean sendWarning(long deltaMailTime, ServiceHandle<T> handle) {
			boolean mailSent = false;
			if (deltaMailTime > SEND_MAIL_INTERVAL) {
				mailSent = true;
				StringBuilder log = new StringBuilder();
				log.append("service address : ").append( handle.getAddress() ).append( " is unavailable, service handle : ").append(handle.toString()).append(" is unavailable.");
				POOL.error(log.toString());
			}
			return mailSent;
		}
		
		private boolean validateHandle(ServiceHandle<T> handle){
			boolean active = false;
			for (int k = 0; k < RETRY_TIMES; k++) {
				if (factory.validateHandle(handle.getAddress())) {
					active = true;
					break;
				}
				try {
					Thread.sleep(RETRY_INTERVAL_MILLIS);
				} catch (InterruptedException e) {
				}
			}
			return active;
		}

		private void cleanInvalidChannel(ServiceHandle<T> handle) {
			Collection<T> invalids = new HashSet<T>(handle.invalidChannels());
			List<T> channels = handle.channels();
			for (T t : channels){
				if (!factory.validateChannel(t)){
					invalids.add(t);
				}
			}
			if (invalids.size()>0){
				channels.removeAll(invalids);
			}
		}
		
		public boolean isRuning() {
			return runing;
		}

		public void setRuning(boolean runing) {
			this.runing = runing;
		}
	}
}
