/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.mapreduce;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.HFileOutputFormat;
import org.apache.hadoop.hbase.mapreduce.PutSortReducer;
import org.apache.hadoop.hbase.mapreduce.TableMapReduceUtil;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Counter;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.mapreduce.lib.input.FileInputFormat;
import org.apache.hadoop.mapreduce.lib.input.TextInputFormat;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.util.GenericOptionsParser;

public class ImportTsv {
    static final String NAME = "importtsv";
    static final String SKIP_LINES_CONF_KEY = "importtsv.skip.bad.lines";
    static final String BULK_OUTPUT_CONF_KEY = "importtsv.bulk.output";
    static final String COLUMNS_CONF_KEY = "importtsv.columns";
    static final String SEPARATOR_CONF_KEY = "importtsv.separator";
    static final String DEFAULT_SEPARATOR = "\t";

    public static Job createSubmittableJob(Configuration conf, String[] args) throws IOException {
        String tableName = args[0];
        Path inputDir = new Path(args[1]);
        Job job = new Job(conf, "importtsv_" + tableName);
        job.setJarByClass(TsvImporter.class);
        FileInputFormat.setInputPaths((Job)job, (Path[])new Path[]{inputDir});
        job.setInputFormatClass(TextInputFormat.class);
        job.setMapperClass(TsvImporter.class);
        String hfileOutPath = conf.get(BULK_OUTPUT_CONF_KEY);
        if (hfileOutPath != null) {
            HTable table = new HTable(conf, tableName);
            job.setReducerClass(PutSortReducer.class);
            Path outputDir = new Path(hfileOutPath);
            FileOutputFormat.setOutputPath((Job)job, (Path)outputDir);
            job.setMapOutputKeyClass(ImmutableBytesWritable.class);
            job.setMapOutputValueClass(Put.class);
            HFileOutputFormat.configureIncrementalLoad(job, table);
        } else {
            TableMapReduceUtil.initTableReducerJob(tableName, null, job);
            job.setNumReduceTasks(0);
        }
        TableMapReduceUtil.addDependencyJars(job);
        return job;
    }

    private static void usage(String errorMsg) {
        if (errorMsg != null && errorMsg.length() > 0) {
            System.err.println("ERROR: " + errorMsg);
        }
        String usage = "Usage: importtsv -Dimporttsv.columns=a,b,c <tablename> <inputdir>\n\nImports the given input directory of TSV data into the specified table.\n\nThe column names of the TSV data must be specified using the -Dimporttsv.columns\noption. This option takes the form of comma-separated column names, where each\ncolumn name is either a simple column family, or a columnfamily:qualifier. The special\ncolumn name HBASE_ROW_KEY is used to designate that this column should be used\nas the row key for each imported record. You must specify exactly one column\nto be the row key.\n\nIn order to prepare data for a bulk data load, pass the option:\n  -Dimporttsv.bulk.output=/path/for/output\n\nOther options that may be specified with -D include:\n  -Dimporttsv.skip.bad.lines=false - fail if encountering an invalid line\n  '-Dimporttsv.separator=|' - eg separate on pipes instead of tabs";
        System.err.println(usage);
    }

    public static void main(String[] args) throws Exception {
        Job job;
        String[] columns;
        Configuration conf = HBaseConfiguration.create();
        String[] otherArgs = new GenericOptionsParser(conf, args).getRemainingArgs();
        if (otherArgs.length < 2) {
            ImportTsv.usage("Wrong number of arguments: " + otherArgs.length);
            System.exit(-1);
        }
        if ((columns = conf.getStrings(COLUMNS_CONF_KEY)) == null) {
            ImportTsv.usage("No columns specified. Please specify with -Dimporttsv.columns=...");
            System.exit(-1);
        }
        int rowkeysFound = 0;
        for (String col : columns) {
            if (!col.equals(TsvParser.ROWKEY_COLUMN_SPEC)) continue;
            ++rowkeysFound;
        }
        if (rowkeysFound != 1) {
            ImportTsv.usage("Must specify exactly one column as " + TsvParser.ROWKEY_COLUMN_SPEC);
            System.exit(-1);
        }
        if (columns.length < 2) {
            ImportTsv.usage("One or more columns in addition to the row key are required");
            System.exit(-1);
        }
        System.exit((job = ImportTsv.createSubmittableJob(conf, otherArgs)).waitForCompletion(true) ? 0 : 1);
    }

    static class TsvImporter
    extends Mapper<LongWritable, Text, ImmutableBytesWritable, Put> {
        private long ts;
        private boolean skipBadLines;
        private Counter badLineCount;
        private TsvParser parser;

        TsvImporter() {
        }

        protected void setup(Mapper.Context context) {
            Configuration conf = context.getConfiguration();
            this.parser = new TsvParser(conf.get(ImportTsv.COLUMNS_CONF_KEY), conf.get(ImportTsv.SEPARATOR_CONF_KEY, ImportTsv.DEFAULT_SEPARATOR));
            if (this.parser.getRowKeyColumnIndex() == -1) {
                throw new RuntimeException("No row key column specified");
            }
            this.ts = System.currentTimeMillis();
            this.skipBadLines = context.getConfiguration().getBoolean(ImportTsv.SKIP_LINES_CONF_KEY, true);
            this.badLineCount = context.getCounter("ImportTsv", "Bad Lines");
        }

        public void map(LongWritable offset, Text value, Mapper.Context context) throws IOException {
            byte[] lineBytes = value.getBytes();
            try {
                TsvParser.ParsedLine parsed = this.parser.parse(lineBytes, value.getLength());
                ImmutableBytesWritable rowKey = new ImmutableBytesWritable(lineBytes, parsed.getRowKeyOffset(), parsed.getRowKeyLength());
                Put put = new Put(rowKey.copyBytes());
                for (int i = 0; i < parsed.getColumnCount(); ++i) {
                    if (i == this.parser.getRowKeyColumnIndex()) continue;
                    KeyValue kv = new KeyValue(lineBytes, parsed.getRowKeyOffset(), parsed.getRowKeyLength(), this.parser.getFamily(i), 0, this.parser.getFamily(i).length, this.parser.getQualifier(i), 0, this.parser.getQualifier(i).length, this.ts, KeyValue.Type.Put, lineBytes, parsed.getColumnOffset(i), parsed.getColumnLength(i));
                    put.add(kv);
                }
                context.write((Object)rowKey, (Object)put);
            }
            catch (TsvParser.BadTsvLineException badLine) {
                if (this.skipBadLines) {
                    System.err.println("Bad line at offset: " + offset.get() + ":\n" + badLine.getMessage());
                    this.badLineCount.increment(1L);
                    return;
                }
                throw new IOException(badLine);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    static class TsvParser {
        private final byte[][] families;
        private final byte[][] qualifiers;
        private final byte separatorByte;
        private int rowKeyColumnIndex;
        public static String ROWKEY_COLUMN_SPEC = "HBASE_ROW_KEY";

        public TsvParser(String columnsSpecification, String separatorStr) {
            byte[] separator = Bytes.toBytes(separatorStr);
            Preconditions.checkArgument((separator.length == 1 ? 1 : 0) != 0, (Object)"TsvParser only supports single-byte separators");
            this.separatorByte = separator[0];
            ArrayList columnStrings = Lists.newArrayList((Iterable)Splitter.on((char)',').trimResults().split((CharSequence)columnsSpecification));
            this.families = new byte[columnStrings.size()][];
            this.qualifiers = new byte[columnStrings.size()][];
            for (int i = 0; i < columnStrings.size(); ++i) {
                String str = (String)columnStrings.get(i);
                if (ROWKEY_COLUMN_SPEC.equals(str)) {
                    this.rowKeyColumnIndex = i;
                    continue;
                }
                String[] parts = str.split(":", 2);
                if (parts.length == 1) {
                    this.families[i] = str.getBytes();
                    this.qualifiers[i] = HConstants.EMPTY_BYTE_ARRAY;
                    continue;
                }
                this.families[i] = parts[0].getBytes();
                this.qualifiers[i] = parts[1].getBytes();
            }
        }

        public int getRowKeyColumnIndex() {
            return this.rowKeyColumnIndex;
        }

        public byte[] getFamily(int idx) {
            return this.families[idx];
        }

        public byte[] getQualifier(int idx) {
            return this.qualifiers[idx];
        }

        public ParsedLine parse(byte[] lineBytes, int length) throws BadTsvLineException {
            ArrayList<Integer> tabOffsets = new ArrayList<Integer>(this.families.length);
            for (int i = 0; i < length; ++i) {
                if (lineBytes[i] != this.separatorByte) continue;
                tabOffsets.add(i);
            }
            if (tabOffsets.isEmpty()) {
                throw new BadTsvLineException("No delimiter");
            }
            tabOffsets.add(length);
            if (tabOffsets.size() > this.families.length) {
                throw new BadTsvLineException("Excessive columns");
            }
            if (tabOffsets.size() <= this.getRowKeyColumnIndex()) {
                throw new BadTsvLineException("No row key");
            }
            return new ParsedLine(tabOffsets, lineBytes);
        }

        public static class BadTsvLineException
        extends Exception {
            private static final long serialVersionUID = 1L;

            public BadTsvLineException(String err) {
                super(err);
            }
        }

        class ParsedLine {
            private final ArrayList<Integer> tabOffsets;
            private byte[] lineBytes;

            ParsedLine(ArrayList<Integer> tabOffsets, byte[] lineBytes) {
                this.tabOffsets = tabOffsets;
                this.lineBytes = lineBytes;
            }

            public int getRowKeyOffset() {
                return this.getColumnOffset(TsvParser.this.rowKeyColumnIndex);
            }

            public int getRowKeyLength() {
                return this.getColumnLength(TsvParser.this.rowKeyColumnIndex);
            }

            public int getColumnOffset(int idx) {
                if (idx > 0) {
                    return this.tabOffsets.get(idx - 1) + 1;
                }
                return 0;
            }

            public int getColumnLength(int idx) {
                return this.tabOffsets.get(idx) - this.getColumnOffset(idx);
            }

            public int getColumnCount() {
                return this.tabOffsets.size();
            }

            public byte[] getLineBytes() {
                return this.lineBytes;
            }
        }
    }
}

