/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.Map;
import java.util.SortedMap;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.DroppedSnapshotException;
import org.apache.hadoop.hbase.RemoteExceptionHandler;
import org.apache.hadoop.hbase.regionserver.FlushRequester;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.HRegionServer;
import org.apache.hadoop.hbase.regionserver.Store;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.util.StringUtils;

class MemStoreFlusher
extends Thread
implements FlushRequester {
    static final Log LOG = LogFactory.getLog(MemStoreFlusher.class);
    private final BlockingQueue<FlushQueueEntry> flushQueue = new DelayQueue<FlushQueueEntry>();
    private final Map<HRegion, FlushQueueEntry> regionsInQueue = new HashMap<HRegion, FlushQueueEntry>();
    private final long threadWakeFrequency;
    private final HRegionServer server;
    private final ReentrantLock lock = new ReentrantLock();
    protected final long globalMemStoreLimit;
    protected final long globalMemStoreLimitLowMark;
    private static final float DEFAULT_UPPER = 0.4f;
    private static final float DEFAULT_LOWER = 0.25f;
    private static final String UPPER_KEY = "hbase.regionserver.global.memstore.upperLimit";
    private static final String LOWER_KEY = "hbase.regionserver.global.memstore.lowerLimit";
    private long blockingStoreFilesNumber;
    private long blockingWaitTime;

    public MemStoreFlusher(Configuration conf, HRegionServer server) {
        this.server = server;
        this.threadWakeFrequency = conf.getLong("hbase.server.thread.wakefrequency", 10000L);
        long max = ManagementFactory.getMemoryMXBean().getHeapMemoryUsage().getMax();
        this.globalMemStoreLimit = MemStoreFlusher.globalMemStoreLimit(max, 0.4f, UPPER_KEY, conf);
        long lower = MemStoreFlusher.globalMemStoreLimit(max, 0.25f, LOWER_KEY, conf);
        if (lower > this.globalMemStoreLimit) {
            lower = this.globalMemStoreLimit;
            LOG.info((Object)"Setting globalMemStoreLimitLowMark == globalMemStoreLimit because supplied hbase.regionserver.global.memstore.lowerLimit was > hbase.regionserver.global.memstore.upperLimit");
        }
        this.globalMemStoreLimitLowMark = lower;
        this.blockingStoreFilesNumber = conf.getInt("hbase.hstore.blockingStoreFiles", -1);
        if (this.blockingStoreFilesNumber == -1L) {
            this.blockingStoreFilesNumber = 1 + conf.getInt("hbase.hstore.compactionThreshold", 3);
        }
        this.blockingWaitTime = conf.getInt("hbase.hstore.blockingWaitTime", 90000);
        LOG.info((Object)("globalMemStoreLimit=" + StringUtils.humanReadableInt((long)this.globalMemStoreLimit) + ", globalMemStoreLimitLowMark=" + StringUtils.humanReadableInt((long)this.globalMemStoreLimitLowMark) + ", maxHeap=" + StringUtils.humanReadableInt((long)max)));
    }

    static long globalMemStoreLimit(long max, float defaultLimit, String key, Configuration c) {
        float limit = c.getFloat(key, defaultLimit);
        return MemStoreFlusher.getMemStoreLimit(max, limit, defaultLimit);
    }

    static long getMemStoreLimit(long max, float limit, float defaultLimit) {
        if (limit >= 0.9f || limit < 0.1f) {
            LOG.warn((Object)("Setting global memstore limit to default of " + defaultLimit + " because supplied value outside allowed range of 0.1 -> 0.9"));
        }
        return (long)((float)max * limit);
    }

    @Override
    public void run() {
        while (!this.server.isStopped()) {
            FlushQueueEntry fqe = null;
            try {
                fqe = this.flushQueue.poll(this.threadWakeFrequency, TimeUnit.MILLISECONDS);
                if (fqe == null || this.flushRegion(fqe)) continue;
                break;
            }
            catch (InterruptedException ex) {
            }
            catch (ConcurrentModificationException ex) {
            }
            catch (Exception ex) {
                LOG.error((Object)("Cache flush failed" + (fqe != null ? " for region " + Bytes.toString(fqe.region.getRegionName()) : "")), (Throwable)ex);
                if (this.server.checkFileSystem()) continue;
                break;
            }
        }
        this.regionsInQueue.clear();
        this.flushQueue.clear();
        LOG.info((Object)(this.getName() + " exiting"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void requestFlush(HRegion r) {
        Map<HRegion, FlushQueueEntry> map = this.regionsInQueue;
        synchronized (map) {
            if (!this.regionsInQueue.containsKey(r)) {
                FlushQueueEntry fqe = new FlushQueueEntry(r);
                this.regionsInQueue.put(r, fqe);
                this.flushQueue.add(fqe);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void interruptIfNecessary() {
        this.lock.lock();
        try {
            this.interrupt();
        }
        finally {
            this.lock.unlock();
        }
    }

    private boolean flushRegion(FlushQueueEntry fqe) {
        HRegion region = fqe.region;
        if (!fqe.region.getRegionInfo().isMetaRegion() && this.isTooManyStoreFiles(region)) {
            if (fqe.isMaximumWait(this.blockingWaitTime)) {
                LOG.info((Object)("Waited " + (System.currentTimeMillis() - fqe.createTime) + "ms on a compaction to clean up 'too many store files'; waited " + "long enough... proceeding with flush of " + region.getRegionNameAsString()));
            } else {
                if (fqe.getRequeueCount() <= 0) {
                    LOG.warn((Object)("Region " + region.getRegionNameAsString() + " has too many " + "store files; delaying flush up to " + this.blockingWaitTime + "ms"));
                }
                this.server.compactSplitThread.requestCompaction(region, this.getName());
                this.flushQueue.add(fqe.requeue(this.blockingWaitTime / 100L));
                return true;
            }
        }
        return this.flushRegion(region, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean flushRegion(HRegion region, boolean emergencyFlush) {
        Map<HRegion, FlushQueueEntry> map = this.regionsInQueue;
        synchronized (map) {
            FlushQueueEntry fqe = this.regionsInQueue.remove(region);
            if (fqe != null && emergencyFlush) {
                this.flushQueue.remove(fqe);
            }
            this.lock.lock();
        }
        try {
            if (region.flushcache()) {
                this.server.compactSplitThread.requestCompaction(region, this.getName());
            }
            this.server.getMetrics().addFlush(region.getRecentFlushInfo());
        }
        catch (DroppedSnapshotException ex) {
            this.server.abort("Replay of HLog required. Forcing server shutdown", ex);
            boolean bl = false;
            return bl;
        }
        catch (IOException ex) {
            LOG.error((Object)("Cache flush failed" + (region != null ? " for region " + Bytes.toString(region.getRegionName()) : "")), (Throwable)RemoteExceptionHandler.checkIOException(ex));
            if (!this.server.checkFileSystem()) {
                boolean bl = false;
                return bl;
            }
        }
        finally {
            this.lock.unlock();
        }
        return true;
    }

    private boolean isTooManyStoreFiles(HRegion region) {
        for (Store hstore : region.stores.values()) {
            if ((long)hstore.getStorefilesCount() <= this.blockingStoreFilesNumber) continue;
            return true;
        }
        return false;
    }

    public synchronized void reclaimMemStoreMemory() {
        if (this.server.getGlobalMemStoreSize() >= this.globalMemStoreLimit) {
            this.flushSomeRegions();
        }
    }

    private synchronized void flushSomeRegions() {
        long globalMemStoreSize = -1L;
        ArrayList<HRegion> regionsToCompact = new ArrayList<HRegion>();
        SortedMap<Long, HRegion> m = this.server.getCopyOfOnlineRegionsSortedBySize();
        while ((globalMemStoreSize = this.server.getGlobalMemStoreSize()) >= this.globalMemStoreLimitLowMark) {
            if (m.size() <= 0) {
                LOG.info((Object)("No online regions to flush though we've been asked flush some; globalMemStoreSize=" + StringUtils.humanReadableInt((long)globalMemStoreSize) + ", globalMemStoreLimitLowMark=" + StringUtils.humanReadableInt((long)this.globalMemStoreLimitLowMark)));
                break;
            }
            HRegion biggestMemStoreRegion = (HRegion)m.remove(m.firstKey());
            LOG.info((Object)("Forced flushing of " + biggestMemStoreRegion.toString() + " because global memstore limit of " + StringUtils.humanReadableInt((long)this.globalMemStoreLimit) + " exceeded; currently " + StringUtils.humanReadableInt((long)globalMemStoreSize) + " and flushing till " + StringUtils.humanReadableInt((long)this.globalMemStoreLimitLowMark)));
            if (!this.flushRegion(biggestMemStoreRegion, true)) {
                LOG.warn((Object)"Flush failed");
                break;
            }
            regionsToCompact.add(biggestMemStoreRegion);
        }
        for (HRegion region : regionsToCompact) {
            this.server.compactSplitThread.requestCompaction(region, this.getName());
        }
    }

    static class FlushQueueEntry
    implements Delayed {
        private final HRegion region;
        private final long createTime;
        private long whenToExpire;
        private int requeueCount = 0;

        FlushQueueEntry(HRegion r) {
            this.region = r;
            this.whenToExpire = this.createTime = System.currentTimeMillis();
        }

        public boolean isMaximumWait(long maximumWait) {
            return System.currentTimeMillis() - this.createTime > maximumWait;
        }

        public int getRequeueCount() {
            return this.requeueCount;
        }

        public FlushQueueEntry requeue(long when) {
            this.whenToExpire = System.currentTimeMillis() + when;
            ++this.requeueCount;
            return this;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return unit.convert(this.whenToExpire - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed other) {
            return Long.valueOf(this.getDelay(TimeUnit.MILLISECONDS) - other.getDelay(TimeUnit.MILLISECONDS)).intValue();
        }
    }
}

