/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Ordering;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryUsage;
import java.nio.ByteBuffer;
import java.text.NumberFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.SortedSet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.io.HalfStoreFileReader;
import org.apache.hadoop.hbase.io.Reference;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.Compression;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileScanner;
import org.apache.hadoop.hbase.io.hfile.LruBlockCache;
import org.apache.hadoop.hbase.regionserver.StoreFileScanner;
import org.apache.hadoop.hbase.regionserver.TimeRangeTracker;
import org.apache.hadoop.hbase.util.BloomFilter;
import org.apache.hadoop.hbase.util.ByteBloomFilter;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Hash;
import org.apache.hadoop.hbase.util.Writables;
import org.apache.hadoop.io.RawComparator;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.WritableUtils;
import org.apache.hadoop.util.StringUtils;

public class StoreFile {
    static final Log LOG = LogFactory.getLog((String)StoreFile.class.getName());
    static final String IO_STOREFILE_BLOOM_ERROR_RATE = "io.storefile.bloom.error.rate";
    static final String IO_STOREFILE_BLOOM_MAX_FOLD = "io.storefile.bloom.max.fold";
    static final String IO_STOREFILE_BLOOM_MAX_KEYS = "io.storefile.bloom.max.keys";
    static final String IO_STOREFILE_BLOOM_ENABLED = "io.storefile.bloom.enabled";
    static final String HFILE_BLOCK_CACHE_SIZE_KEY = "hfile.block.cache.size";
    public static final byte[] MAX_SEQ_ID_KEY = Bytes.toBytes("MAX_SEQ_ID_KEY");
    public static final byte[] MAJOR_COMPACTION_KEY = Bytes.toBytes("MAJOR_COMPACTION_KEY");
    static final byte[] BLOOM_FILTER_TYPE_KEY = Bytes.toBytes("BLOOM_FILTER_TYPE");
    static final byte[] TIMERANGE_KEY = Bytes.toBytes("TIMERANGE");
    static final String BLOOM_FILTER_META_KEY = "BLOOM_FILTER_META";
    static final String BLOOM_FILTER_DATA_KEY = "BLOOM_FILTER_DATA";
    public static final int DEFAULT_BLOCKSIZE_SMALL = 8192;
    private static BlockCache hfileBlockCache = null;
    private final FileSystem fs;
    private final Path path;
    private Reference reference;
    private Path referencePath;
    private boolean blockcache;
    private boolean inMemory;
    private long sequenceid = -1L;
    private AtomicBoolean majorCompaction = null;
    public static final byte[] BULKLOAD_TASK_KEY = Bytes.toBytes("BULKLOAD_SOURCE_TASK");
    public static final byte[] BULKLOAD_TIME_KEY = Bytes.toBytes("BULKLOAD_TIMESTAMP");
    private Map<byte[], byte[]> metadataMap;
    private static final Pattern REF_NAME_PARSER = Pattern.compile("^(\\d+)(?:\\.(.+))?$");
    private volatile Reader reader;
    private static final Random rand = new Random();
    private final Configuration conf;
    private final BloomType bloomType;

    StoreFile(FileSystem fs, Path p, boolean blockcache, Configuration conf, BloomType bt, boolean inMemory) throws IOException {
        this.conf = conf;
        this.fs = fs;
        this.path = p;
        this.blockcache = blockcache;
        this.inMemory = inMemory;
        if (StoreFile.isReference(p)) {
            this.reference = Reference.read(fs, p);
            this.referencePath = StoreFile.getReferredToFile(this.path);
        }
        if (conf.getBoolean(IO_STOREFILE_BLOOM_ENABLED, true)) {
            this.bloomType = bt;
        } else {
            this.bloomType = BloomType.NONE;
            LOG.info((Object)"Ignoring bloom filter check for file (disabled in config)");
        }
    }

    Path getPath() {
        return this.path;
    }

    byte[] getFamily() {
        return Bytes.toBytes(this.path.getParent().getName());
    }

    boolean isReference() {
        return this.reference != null;
    }

    public static boolean isReference(Path p) {
        return !p.getName().startsWith("_") && StoreFile.isReference(p, REF_NAME_PARSER.matcher(p.getName()));
    }

    public static boolean isReference(Path p, Matcher m) {
        if (m == null || !m.matches()) {
            LOG.warn((Object)("Failed match of store file name " + p.toString()));
            throw new RuntimeException("Failed match of store file name " + p.toString());
        }
        return m.groupCount() > 1 && m.group(2) != null;
    }

    static Path getReferredToFile(Path p) {
        Matcher m = REF_NAME_PARSER.matcher(p.getName());
        if (m == null || !m.matches()) {
            LOG.warn((Object)("Failed match of store file name " + p.toString()));
            throw new RuntimeException("Failed match of store file name " + p.toString());
        }
        String otherRegion = m.group(2);
        Path tableDir = p.getParent().getParent().getParent();
        String nameStrippedOfSuffix = m.group(1);
        return new Path(new Path(new Path(tableDir, otherRegion), p.getParent().getName()), nameStrippedOfSuffix);
    }

    boolean isMajorCompaction() {
        if (this.majorCompaction == null) {
            throw new NullPointerException("This has not been set yet");
        }
        return this.majorCompaction.get();
    }

    public long getMaxSequenceId() {
        return this.sequenceid;
    }

    public static long getMaxSequenceIdInList(List<StoreFile> sfs) {
        long max = 0L;
        for (StoreFile sf : sfs) {
            if (sf.isBulkLoadResult()) continue;
            max = Math.max(max, sf.getMaxSequenceId());
        }
        return max;
    }

    boolean isBulkLoadResult() {
        return this.metadataMap.containsKey(BULKLOAD_TIME_KEY);
    }

    public long getBulkLoadTimestamp() {
        return Bytes.toLong(this.metadataMap.get(BULKLOAD_TIME_KEY));
    }

    public static synchronized BlockCache getBlockCache(Configuration conf) {
        if (hfileBlockCache != null) {
            return hfileBlockCache;
        }
        float cachePercentage = conf.getFloat(HFILE_BLOCK_CACHE_SIZE_KEY, 0.0f);
        if (cachePercentage == 0.0f) {
            return null;
        }
        if ((double)cachePercentage > 1.0) {
            throw new IllegalArgumentException("hfile.block.cache.size must be between 0.0 and 1.0, not > 1.0");
        }
        MemoryUsage mu = ManagementFactory.getMemoryMXBean().getHeapMemoryUsage();
        long cacheSize = (long)((float)mu.getMax() * cachePercentage);
        LOG.info((Object)("Allocating LruBlockCache with maximum size " + StringUtils.humanReadableInt((long)cacheSize)));
        hfileBlockCache = new LruBlockCache(cacheSize, 8192L);
        return hfileBlockCache;
    }

    public BlockCache getBlockCache() {
        return this.blockcache ? StoreFile.getBlockCache(this.conf) : null;
    }

    private Reader open() throws IOException {
        if (this.reader != null) {
            throw new IllegalAccessError("Already open");
        }
        this.reader = this.isReference() ? new HalfStoreFileReader(this.fs, this.referencePath, this.getBlockCache(), this.reference) : new Reader(this.fs, this.path, this.getBlockCache(), this.inMemory);
        this.metadataMap = Collections.unmodifiableMap(this.reader.loadFileInfo());
        byte[] b = this.metadataMap.get(MAX_SEQ_ID_KEY);
        if (b != null) {
            this.sequenceid = Bytes.toLong(b);
            if (this.isReference() && Reference.isTopFileRegion(this.reference.getFileRegion())) {
                ++this.sequenceid;
            }
        }
        this.reader.setSequenceID(this.sequenceid);
        b = this.metadataMap.get(MAJOR_COMPACTION_KEY);
        if (b != null) {
            boolean mc = Bytes.toBoolean(b);
            if (this.majorCompaction == null) {
                this.majorCompaction = new AtomicBoolean(mc);
            } else {
                this.majorCompaction.set(mc);
            }
        } else {
            this.majorCompaction = new AtomicBoolean(false);
        }
        if (this.bloomType != BloomType.NONE) {
            this.reader.loadBloomfilter();
        }
        try {
            byte[] timerangeBytes = this.metadataMap.get(TIMERANGE_KEY);
            if (timerangeBytes != null) {
                this.reader.timeRangeTracker = new TimeRangeTracker();
                Writables.copyWritable(timerangeBytes, (Writable)this.reader.timeRangeTracker);
            }
        }
        catch (IllegalArgumentException e) {
            LOG.error((Object)"Error reading timestamp range data from meta -- proceeding without", (Throwable)e);
            this.reader.timeRangeTracker = null;
        }
        return this.reader;
    }

    public Reader createReader() throws IOException {
        if (this.reader == null) {
            this.reader = this.open();
        }
        return this.reader;
    }

    public Reader getReader() {
        return this.reader;
    }

    public synchronized void closeReader() throws IOException {
        if (this.reader != null) {
            this.reader.close();
            this.reader = null;
        }
    }

    public void deleteReader() throws IOException {
        this.closeReader();
        this.fs.delete(this.getPath(), true);
    }

    public String toString() {
        return this.path.toString() + (this.isReference() ? "-" + this.referencePath + "-" + this.reference.toString() : "");
    }

    public String toStringDetailed() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.path.toString());
        sb.append(", isReference=").append(this.isReference());
        sb.append(", isBulkLoadResult=").append(this.isBulkLoadResult());
        if (this.isBulkLoadResult()) {
            sb.append(", bulkLoadTS=").append(this.getBulkLoadTimestamp());
        } else {
            sb.append(", seqid=").append(this.getMaxSequenceId());
        }
        sb.append(", majorCompaction=").append(this.isMajorCompaction());
        return sb.toString();
    }

    public static Path rename(FileSystem fs, Path src, Path tgt) throws IOException {
        if (!fs.exists(src)) {
            throw new FileNotFoundException(src.toString());
        }
        if (!fs.rename(src, tgt)) {
            throw new IOException("Failed rename of " + src + " to " + tgt);
        }
        return tgt;
    }

    public static Writer createWriter(FileSystem fs, Path dir, int blocksize) throws IOException {
        return StoreFile.createWriter(fs, dir, blocksize, null, null, null, BloomType.NONE, 0);
    }

    public static Writer createWriter(FileSystem fs, Path dir, int blocksize, Compression.Algorithm algorithm, KeyValue.KVComparator c, Configuration conf, BloomType bloomType, int maxKeySize) throws IOException {
        if (!fs.exists(dir)) {
            fs.mkdirs(dir);
        }
        Path path = StoreFile.getUniqueFile(fs, dir);
        if (conf == null || !conf.getBoolean(IO_STOREFILE_BLOOM_ENABLED, true)) {
            bloomType = BloomType.NONE;
        }
        return new Writer(fs, path, blocksize, algorithm == null ? HFile.DEFAULT_COMPRESSION_ALGORITHM : algorithm, conf, c == null ? KeyValue.COMPARATOR : c, bloomType, maxKeySize);
    }

    public static Path getUniqueFile(FileSystem fs, Path dir) throws IOException {
        if (!fs.getFileStatus(dir).isDir()) {
            throw new IOException("Expecting " + dir.toString() + " to be a directory");
        }
        return fs.getFileStatus(dir).isDir() ? StoreFile.getRandomFilename(fs, dir) : dir;
    }

    static Path getRandomFilename(FileSystem fs, Path dir) throws IOException {
        return StoreFile.getRandomFilename(fs, dir, null);
    }

    static Path getRandomFilename(FileSystem fs, Path dir, String suffix) throws IOException {
        long id = -1L;
        Path p = null;
        while (fs.exists(p = new Path(dir, Long.toString(id = Math.abs(rand.nextLong())) + (suffix == null || suffix.length() <= 0 ? "" : suffix)))) {
        }
        return p;
    }

    static Path split(FileSystem fs, Path splitDir, StoreFile f, byte[] splitRow, Reference.Range range) throws IOException {
        Reference r = new Reference(splitRow, range);
        String parentRegionName = f.getPath().getParent().getParent().getName();
        Path p = new Path(splitDir, f.getPath().getName() + "." + parentRegionName);
        return r.write(fs, p);
    }

    static abstract class Comparators {
        static final Comparator<StoreFile> FLUSH_TIME = Ordering.compound((Iterable)ImmutableList.of((Object)Ordering.natural().onResultOf((Function)new GetBulkTime()), (Object)Ordering.natural().onResultOf((Function)new GetSeqId()), (Object)Ordering.natural().onResultOf((Function)new GetPathName())));

        Comparators() {
        }

        private static class GetPathName
        implements Function<StoreFile, String> {
            private GetPathName() {
            }

            public String apply(StoreFile sf) {
                return sf.getPath().getName();
            }
        }

        private static class GetSeqId
        implements Function<StoreFile, Long> {
            private GetSeqId() {
            }

            public Long apply(StoreFile sf) {
                if (sf.isBulkLoadResult()) {
                    return -1L;
                }
                return sf.getMaxSequenceId();
            }
        }

        private static class GetBulkTime
        implements Function<StoreFile, Long> {
            private GetBulkTime() {
            }

            public Long apply(StoreFile sf) {
                if (!sf.isBulkLoadResult()) {
                    return Long.MAX_VALUE;
                }
                return sf.getBulkLoadTimestamp();
            }
        }
    }

    public static class Reader {
        static final Log LOG = LogFactory.getLog((String)Reader.class.getName());
        protected BloomFilter bloomFilter = null;
        protected BloomType bloomFilterType;
        private final HFile.Reader reader;
        protected TimeRangeTracker timeRangeTracker = null;
        protected long sequenceID = -1L;

        public Reader(FileSystem fs, Path path, BlockCache blockCache, boolean inMemory) throws IOException {
            this.reader = new HFile.Reader(fs, path, blockCache, inMemory);
            this.bloomFilterType = BloomType.NONE;
        }

        public RawComparator<byte[]> getComparator() {
            return this.reader.getComparator();
        }

        public StoreFileScanner getStoreFileScanner(boolean cacheBlocks, boolean pread) {
            return new StoreFileScanner(this, this.getScanner(cacheBlocks, pread));
        }

        @Deprecated
        public HFileScanner getScanner(boolean cacheBlocks, boolean pread) {
            return this.reader.getScanner(cacheBlocks, pread);
        }

        public void close() throws IOException {
            this.reader.close();
        }

        public boolean shouldSeek(Scan scan, SortedSet<byte[]> columns) {
            return this.passesTimerangeFilter(scan) && this.passesBloomFilter(scan, columns);
        }

        private boolean passesTimerangeFilter(Scan scan) {
            if (this.timeRangeTracker == null) {
                return true;
            }
            return this.timeRangeTracker.includesTimeRange(scan.getTimeRange());
        }

        private boolean passesBloomFilter(Scan scan, SortedSet<byte[]> columns) {
            byte[] key;
            if (this.bloomFilter == null || !scan.isGetScan()) {
                return true;
            }
            byte[] row = scan.getStartRow();
            switch (this.bloomFilterType) {
                case ROW: {
                    key = row;
                    break;
                }
                case ROWCOL: {
                    if (columns != null && columns.size() == 1) {
                        byte[] col = columns.first();
                        key = Bytes.add(row, col);
                        break;
                    }
                }
                default: {
                    return true;
                }
            }
            try {
                ByteBuffer bloom = this.reader.getMetaBlock(StoreFile.BLOOM_FILTER_DATA_KEY, true);
                if (bloom != null) {
                    if (this.bloomFilterType == BloomType.ROWCOL) {
                        return this.bloomFilter.contains(key, bloom) || this.bloomFilter.contains(row, bloom);
                    }
                    return this.bloomFilter.contains(key, bloom);
                }
            }
            catch (IOException e) {
                LOG.error((Object)"Error reading bloom filter data -- proceeding without", (Throwable)e);
                this.setBloomFilterFaulty();
            }
            catch (IllegalArgumentException e) {
                LOG.error((Object)"Bad bloom filter data -- proceeding without", (Throwable)e);
                this.setBloomFilterFaulty();
            }
            return true;
        }

        public Map<byte[], byte[]> loadFileInfo() throws IOException {
            Map<byte[], byte[]> fi = this.reader.loadFileInfo();
            byte[] b = fi.get(BLOOM_FILTER_TYPE_KEY);
            if (b != null) {
                this.bloomFilterType = BloomType.valueOf(Bytes.toString(b));
            }
            return fi;
        }

        public void loadBloomfilter() {
            if (this.bloomFilter != null) {
                return;
            }
            try {
                ByteBuffer b = this.reader.getMetaBlock(StoreFile.BLOOM_FILTER_META_KEY, false);
                if (b != null) {
                    if (this.bloomFilterType == BloomType.NONE) {
                        throw new IOException("valid bloom filter type not found in FileInfo");
                    }
                    this.bloomFilter = new ByteBloomFilter(b);
                    LOG.info((Object)("Loaded " + (this.bloomFilterType == BloomType.ROW ? "row" : "col") + " bloom filter metadata for " + this.reader.getName()));
                }
            }
            catch (IOException e) {
                LOG.error((Object)"Error reading bloom filter meta -- proceeding without", (Throwable)e);
                this.bloomFilter = null;
            }
            catch (IllegalArgumentException e) {
                LOG.error((Object)"Bad bloom filter meta -- proceeding without", (Throwable)e);
                this.bloomFilter = null;
            }
        }

        public int getFilterEntries() {
            return this.bloomFilter != null ? this.bloomFilter.getKeyCount() : this.reader.getFilterEntries();
        }

        public ByteBuffer getMetaBlock(String bloomFilterDataKey, boolean cacheBlock) throws IOException {
            return this.reader.getMetaBlock(bloomFilterDataKey, cacheBlock);
        }

        public void setBloomFilterFaulty() {
            this.bloomFilter = null;
        }

        public byte[] getLastKey() {
            return this.reader.getLastKey();
        }

        public byte[] midkey() throws IOException {
            return this.reader.midkey();
        }

        public long length() {
            return this.reader.length();
        }

        public int getEntries() {
            return this.reader.getEntries();
        }

        public byte[] getFirstKey() {
            return this.reader.getFirstKey();
        }

        public long indexSize() {
            return this.reader.indexSize();
        }

        public BloomType getBloomFilterType() {
            return this.bloomFilterType;
        }

        public long getSequenceID() {
            return this.sequenceID;
        }

        public void setSequenceID(long sequenceID) {
            this.sequenceID = sequenceID;
        }
    }

    public static class Writer {
        private final BloomFilter bloomFilter;
        private final BloomType bloomType;
        private KeyValue.KVComparator kvComparator;
        private KeyValue lastKv = null;
        private byte[] lastByteArray = null;
        TimeRangeTracker timeRangeTracker = new TimeRangeTracker();
        boolean isTimeRangeTrackerSet = false;
        protected HFile.Writer writer;

        public Writer(FileSystem fs, Path path, int blocksize, Compression.Algorithm compress, Configuration conf, KeyValue.KVComparator comparator, BloomType bloomType, int maxKeys) throws IOException {
            this.writer = new HFile.Writer(fs, path, blocksize, compress, comparator.getRawComparator());
            this.kvComparator = comparator;
            ByteBloomFilter bloom = null;
            BloomType bt = BloomType.NONE;
            if (bloomType != BloomType.NONE && conf != null) {
                float err = conf.getFloat(StoreFile.IO_STOREFILE_BLOOM_ERROR_RATE, 0.01f);
                if (bloomType == BloomType.ROWCOL) {
                    err /= 2.0f;
                }
                int maxFold = conf.getInt(StoreFile.IO_STOREFILE_BLOOM_MAX_FOLD, 7);
                int tooBig = conf.getInt(StoreFile.IO_STOREFILE_BLOOM_MAX_KEYS, 128000000);
                if (maxKeys < tooBig) {
                    try {
                        bloom = new ByteBloomFilter(maxKeys, err, Hash.getHashType(conf), maxFold);
                        bloom.allocBloom();
                        bt = bloomType;
                    }
                    catch (IllegalArgumentException iae) {
                        LOG.warn((Object)String.format("Parse error while creating bloom for %s (%d, %f)", path, maxKeys, Float.valueOf(err)), (Throwable)iae);
                        bloom = null;
                        bt = BloomType.NONE;
                    }
                } else if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("Skipping bloom filter because max keysize too large: " + maxKeys));
                }
            }
            this.bloomFilter = bloom;
            this.bloomType = bt;
        }

        public void appendMetadata(long maxSequenceId, boolean majorCompaction) throws IOException {
            this.writer.appendFileInfo(MAX_SEQ_ID_KEY, Bytes.toBytes(maxSequenceId));
            this.writer.appendFileInfo(MAJOR_COMPACTION_KEY, Bytes.toBytes(majorCompaction));
            this.appendTimeRangeMetadata();
        }

        public void appendTimeRangeMetadata() throws IOException {
            this.appendFileInfo(TIMERANGE_KEY, WritableUtils.toByteArray((Writable[])new Writable[]{this.timeRangeTracker}));
        }

        public void setTimeRangeTracker(TimeRangeTracker trt) {
            this.timeRangeTracker = trt;
            this.isTimeRangeTrackerSet = true;
        }

        public void includeInTimeRangeTracker(KeyValue kv) {
            if (!this.isTimeRangeTrackerSet) {
                this.timeRangeTracker.includeTimestamp(kv);
            }
        }

        public void includeInTimeRangeTracker(byte[] key) {
            if (!this.isTimeRangeTrackerSet) {
                this.timeRangeTracker.includeTimestamp(key);
            }
        }

        public void append(KeyValue kv) throws IOException {
            if (this.bloomFilter != null) {
                boolean newKey = true;
                if (this.lastKv != null) {
                    switch (this.bloomType) {
                        case ROW: {
                            newKey = !this.kvComparator.matchingRows(kv, this.lastKv);
                            break;
                        }
                        case ROWCOL: {
                            newKey = !this.kvComparator.matchingRowColumn(kv, this.lastKv);
                            break;
                        }
                        case NONE: {
                            newKey = false;
                        }
                    }
                }
                if (newKey) {
                    switch (this.bloomType) {
                        case ROW: {
                            this.bloomFilter.add(kv.getBuffer(), kv.getRowOffset(), kv.getRowLength());
                            break;
                        }
                        case ROWCOL: {
                            int ro = kv.getRowOffset();
                            short rl = kv.getRowLength();
                            int qo = kv.getQualifierOffset();
                            int ql = kv.getQualifierLength();
                            byte[] result = new byte[rl + ql];
                            System.arraycopy(kv.getBuffer(), ro, result, 0, rl);
                            System.arraycopy(kv.getBuffer(), qo, result, rl, ql);
                            this.bloomFilter.add(result);
                            break;
                        }
                    }
                    this.lastKv = kv;
                }
            }
            this.writer.append(kv);
            this.includeInTimeRangeTracker(kv);
        }

        public Path getPath() {
            return this.writer.getPath();
        }

        boolean hasBloom() {
            return this.bloomFilter != null;
        }

        public void append(byte[] key, byte[] value) throws IOException {
            if (!(this.bloomFilter == null || this.lastByteArray != null && Arrays.equals(key, this.lastByteArray))) {
                this.bloomFilter.add(key);
                this.lastByteArray = key;
            }
            this.writer.append(key, value);
            this.includeInTimeRangeTracker(key);
        }

        public void close() throws IOException {
            if (this.bloomFilter != null && this.bloomFilter.getKeyCount() > 0) {
                this.bloomFilter.compactBloom();
                if (this.bloomFilter.getMaxKeys() > 0) {
                    int b = this.bloomFilter.getByteSize();
                    int k = this.bloomFilter.getKeyCount();
                    int m = this.bloomFilter.getMaxKeys();
                    LOG.info((Object)("Bloom added to HFile.  " + b + "B, " + k + "/" + m + " (" + NumberFormat.getPercentInstance().format((double)k / (double)m) + ")"));
                }
                this.writer.appendMetaBlock(StoreFile.BLOOM_FILTER_META_KEY, this.bloomFilter.getMetaWriter());
                this.writer.appendMetaBlock(StoreFile.BLOOM_FILTER_DATA_KEY, this.bloomFilter.getDataWriter());
                this.writer.appendFileInfo(BLOOM_FILTER_TYPE_KEY, Bytes.toBytes(this.bloomType.toString()));
            }
            this.writer.close();
        }

        public void appendFileInfo(byte[] key, byte[] value) throws IOException {
            this.writer.appendFileInfo(key, value);
        }
    }

    public static enum BloomType {
        NONE,
        ROW,
        ROWCOL;

    }
}

