/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.zookeeper.version.util;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

public class VerGen {
    private static final String PACKAGE_NAME = "org.apache.zookeeper.version";
    private static final String TYPE_NAME = "Info";

    static void printUsage() {
        System.out.print("Usage:\tjava  -cp <classpath> org.apache.zookeeper."
                + "version.util.VerGen maj.min.micro rev buildDate");
        System.exit(1);
    }

    static void generateFile(File outputDir, int maj, int min, int micro, int rev,
            String buildDate) {
        String path = PACKAGE_NAME.replaceAll("\\.", "/");
        File pkgdir = new File(outputDir, path);
        if (!pkgdir.exists()) {
            // create the pkg directory
            boolean ret = pkgdir.mkdirs();
            if (!ret) {
                System.out.println("Cannnot create directory: " + path);
                System.exit(1);
            }
        } else if (!pkgdir.isDirectory()) {
            // not a directory
            System.out.println(path + " is not a directory.");
            System.exit(1);
        }
        File file = new File(pkgdir, TYPE_NAME + ".java");
        FileWriter w = null;
        try {
            w = new FileWriter(file);
            w.write("// Do not edit!\n// File generated by org.apache.zookeeper"
                    + ".version.util.VerGen.\n");
            w.write("package " + PACKAGE_NAME + ";\n\n");
            w.write("public interface " + TYPE_NAME + " {\n");
            w.write("    public static final int MAJOR=" + maj + ";\n");
            w.write("    public static final int MINOR=" + min + ";\n");
            w.write("    public static final int MICRO=" + micro + ";\n");
            if (rev < 0) {
                System.out.println("Unknown REVISION number, using " + rev);
            }
            w.write("    public static final int REVISION=" + rev + ";\n");
            w.write("    public static final String BUILD_DATE=\"" + buildDate
                    + "\";\n");
            w.write("}\n");
        } catch (IOException e) {
            System.out.println("Unable to generate version.Info file: "
                    + e.getMessage());
            System.exit(1);
        } finally {
            if (w != null) {
                try {
                    w.close();
                } catch (IOException e) {
                    System.out.println("Unable to close file writer"
                            + e.getMessage());
                }
            }
        }
    }

    /**
     * Emits a org.apache.zookeeper.version.Info interface file with version and
     * revision information constants set to the values passed in as command
     * line parameters. The file is created in the current directory. <br>
     * Usage: java org.apache.zookeeper.version.util.VerGen maj.min.micro rev
     * buildDate
     *
     * @param args
     *            <ul>
     *            <li>maj - major version number
     *            <li>min - minor version number
     *            <li>micro - minor minor version number
     *            <li>rev - current SVN revision number
     *            <li>buildDate - date the build
     *            </ul>
     */
    public static void main(String[] args) {
        if (args.length != 3)
            printUsage();
        try {
            String[] v = args[0].split("\\.");
            if (v.length != 3) {
                System.err
                        .println("Invalid version number format, must be \"x.y.z\"");
                System.exit(1);
            }
            int maj = Integer.parseInt(v[0]);
            int min = Integer.parseInt(v[1]);
            int micro = Integer.parseInt(v[2]);
            int rev;
            try {
                rev = Integer.parseInt(args[1]);
            } catch (NumberFormatException e) {
                rev = -1;
            }
            generateFile(new File("."), maj, min, micro, rev, args[2]);
        } catch (NumberFormatException e) {
            System.err
                .println("All version-related parameters must be valid integers!");
            throw e;
        }
    }

}
